<?php

namespace App\Services\Units;

use App\Core\Exception\ApiException;
use App\Core\File\FileRemover\FileRemoverInterface;
use App\Core\File\FileToUpload\FileToUploadInterface;
use App\Core\File\FileUploader\FileUploaderInterface;
use App\Core\Transactions\TransactionInterface;
use App\Entity\Images\Image;
use App\Entity\Units\Unit;
use App\Entity\Units\UnitsInterface;
use App\Models\Units\UnitModel;
use Symfony\Contracts\Translation\TranslatorInterface;

class UpdateUnit
{
    public function __construct(
        private TransactionInterface $transaction,
        private FileRemoverInterface $fileRemover,
        private FileUploaderInterface $fileUploader,
        private UnitsInterface $units,
        private TranslatorInterface $translator
    ) {}

    public function update(Unit $unit, UnitModel $unitModel): void
    {
        $sentUnitChildren = $unitModel->getUnitChildren();
        $currentUnitChildren = $unit->getUnitChildrenIDs();

        if (!$unit->getType()->isEmergency() && !empty($sentUnitChildren))
        {
            throw new ApiException($this->translator->trans('unit.unitChildren.type'));
        }

        $toAdd = array_diff($sentUnitChildren,$currentUnitChildren);
        $toRemove = array_diff($currentUnitChildren,$sentUnitChildren);

        $this->transaction->begin();

        $unit->updateTitle($unitModel->getTitle());
        $unit->updateDescription($unitModel->getDescription());

        if ($unitModel->isBanner()) {
            $this->replaceImage($unit->getBanner(), $unitModel->getBanner());
            $unit->getBanner()->updateImage(
                $unitModel->getBanner()->getUniqueName()
            );
        }

        if ($unitModel->isPhoto()) {
            $this->replaceImage($unit->getPhoto(), $unitModel->getPhoto());
            $unit->getPhoto()->updateImage(
                $unitModel->getPhoto()->getUniqueName()
            );
        }

        $unitsChildrenToAdd = $this->units->findAllByIDs($toAdd);
        foreach ($unitsChildrenToAdd as $unitChildren)
        {
            $unit->addUnitChildren($unitChildren);
        }

        $unitChildrenToRemove = $this->units->findAllByIDs($toRemove);
        foreach ($unitChildrenToRemove as $unitChildren)
        {
            $unit->removeUnitChildren($unitChildren);
        }

        $this->transaction->end();
    }

    private function replaceImage(Image $image, FileToUploadInterface $fileToUpload): void
    {
        $this->fileRemover->remove($image->getImage());
        $this->fileUploader->upload($fileToUpload);
    }
}