<?php

namespace App\Services\Reports;

use App\Adapter\Reports\Email\ReportNotificationsInterface;
use App\Core\Exception\ApiException;
use App\Core\Transactions\TransactionInterface;
use App\Entity\Posts\PostsInterface;
use App\Entity\Reports\ReadModel\QuarterReportRecipientDTO;
use App\Entity\Reports\ReadModel\ReportsQueryInterface;
use App\Entity\Reports\Report;
use App\Entity\Reports\ReportsInterface;
use App\Entity\Units\UnitsInterface;
use App\Models\Reports\RecipientReportModel;
use App\Models\Reports\ReportModel;
use Symfony\Contracts\Translation\TranslatorInterface;

class CreateQuarterReport
{
    public function __construct(
        private TranslatorInterface $translator,
        private TransactionInterface $transaction,
        private UnitsInterface $units,
        private PostsInterface $posts,
        private ReportsInterface $reports,
        private ReportsQueryInterface $reportsQuery,
        private ReportNotificationsInterface $reportNotifications
    )
    {
    }

    /**
     * @throws ApiException
     */
    public function create(ReportModel $reportModel)
    {
        $report = $this->createReport($reportModel);

        $recipients = $this->reportsQuery->getQuarterReportRecipients($report);


        $this->transaction->begin();

        $this->reports->add($report);

        foreach ($recipients as $recipient) {
            $this->reportNotifications->sendQuarterReportEmail($report, $recipient);
        }
        $this->transaction->end();
    }

    /**
     * @throws ApiException
     */
    public function preview(RecipientReportModel $recipientReportModel)
    {
        $report = $this->createReport($recipientReportModel);

        $this->reportNotifications->sendQuarterReportEmail($report, $recipientReportModel->getQuarterReportRecipientDTO());
    }

    private function createReport(ReportModel $reportModel): Report
    {
        $unit = $this->units->findOneByUnitID($reportModel->getUnitId());

        $post = $this->posts->findOneByPostIDAndUnitID($reportModel->getPostId(), $reportModel->getUnitId());

        if (!$post)
            throw new ApiException($this->translator->trans('post.invalid_unit'));

        return new Report(
            $reportModel->getReportName(),
            $unit,
            $post,
            $reportModel->getDateStartObject(),
            $reportModel->getDateEndObject(),
            $reportModel->getDescription()
        );
    }
}