<?php

namespace App\PaymentSystem;

use App\Entity\Orders\Order;
use App\PaymentSystem\Api\Api;
use App\PaymentSystem\Api\Request\ApiRequest;

class PaymentTransaction extends ApiRequest
{
    public const LANGUAGE_PL = 'pl';
    public const LANGUAGE_EN = 'en';
    public const LANGUAGE_DE = 'de';
    public const LANGUAGE_ES = 'es';
    public const LANGUAGE_IT = 'it';

    public const CHANNEL_CARD = 1;
    public const CHANNEL_WIRE = 2;
    public const CHANNEL_TRADITIONAL_WIRE = 4;
    public const CHANNEL_ALL_24_7 = 16;
    public const CHANNEL_PREPAYMENT = 32;
    public const CHANNEL_PAY_BY_LINK = 64;

    public const ENCODING_ISO_8859_2 = 'ISO-8859-2';
    public const ENCODING_UTF_8 = 'UTF-8';
    public const ENCODING_WINDOWS_1250 = 'Windows-1250';

    /**
     * @var array
     */
    protected $signatureAttributes = [
        'sessionId',
        'merchantId',
        'amount',
        'currency',
        'crc',
    ];


    /**
     * @param array $parameters
     */
    public function __construct(array $parameters = [])
    {
        $this->parameters = array_merge([
            'currency' => 'PLN',
            'country' => 'PL',
            'language' => self::LANGUAGE_PL,
        ], $parameters);
    }

    public static function fromOrder(Order $order): PaymentTransaction
    {
        return new PaymentTransaction(
            [
                'sessionId' => (string)$order->getUuid(),
                'urlReturn' => $order->getUrlReturn(),
                'urlStatus' => $order->getUrlWebhook(),
                'amount' => $order->getAmount(),
                'email' => $order->getEmail(),
                'description' => $order->getDescription(),
                'transferLabel' => 'Darowizna salvatti.edu.pl'
            ]
        );
    }

    /**
     * @param string $methodRefId
     * @return $this
     */
    public function addMethodRefId(string $methodRefId): PaymentTransaction
    {
        $this->parameters['methodRefId'] = $methodRefId;
        return $this;
    }

    /**
     * @param int $channel
     * @return $this
     */
    public function addChannel(int $channel): PaymentTransaction
    {
        $this->parameters['channel'] = $channel;
        return $this;
    }

    /**
     * @param string $urlCardNotification
     * @return $this
     */
    public function addUrlCardNotification(string $urlCardNotification): PaymentTransaction
    {
        $this->parameters['urlCardPaymentNotification'] = $urlCardNotification;
        return $this;
    }
}
