<?php

namespace App\PaymentSystem\Api;

use App\PaymentSystem\Api\Request\ParametersApiRequest;
use App\PaymentSystem\Api\Request\SignedApiRequest;
use App\PaymentSystem\Api\Response\CardChargeResponse;
use App\PaymentSystem\Api\Response\InfoTransactionResponse;
use App\PaymentSystem\Api\Response\RegisterTransactionResponse;
use App\PaymentSystem\Api\Response\TestConnectionResponse;
use App\PaymentSystem\Api\Response\VerifyTransactionResponse;
use App\PaymentSystem\Config;
use App\PaymentSystem\Exceptions\ApiResponseException;
use GuzzleHttp\Client;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\RequestException;
use Psr\Http\Message\ResponseInterface;

class Api
{
    public const VERSION = '3.2';

    public const URL_LIVE = 'https://secure.przelewy24.pl/';
    public const URL_SANDBOX = 'https://sandbox.przelewy24.pl/';

    public const API_VERSION = 'api/v1/';

    public const ENDPOINT_TEST = 'testAccess';
    public const ENDPOINT_REGISTER = 'transaction/register';
    public const ENDPOINT_VERIFY = 'transaction/verify';
    public const ENDPOINT_PAYMENT_GATEWAY = 'trnRequest';
    public const ENDPOINT_CARD_CHARGE = 'card/charge';
    public const ENDPOINT_INFO = 'transaction/by/sessionId/';

    /**
     * @var Config
     */
    private $config;

    /**
     * @var ClientInterface
     */
    private $client;

    /**
     * @param Config $config
     */
    public function __construct(Config $config)
    {
        $this->config = $config;
    }

    /**
     * @return TestConnectionResponse
     * @throws ApiResponseException
     */
    public function testConnection(): TestConnectionResponse
    {
        $response = $this->client()->get(self::ENDPOINT_TEST, []);

        return new TestConnectionResponse($response);
    }

    /**
     * @param SignedApiRequest $apiRequest
     * @return RegisterTransactionResponse
     * @throws ApiResponseException
     */
    public function registerTransaction(SignedApiRequest $apiRequest): RegisterTransactionResponse
    {
        $apiRequest->setConfig($this->config);

        $response = new RegisterTransactionResponse(
            $this->request(self::ENDPOINT_REGISTER, $apiRequest->parameters())
        );

        $response->setGatewayUrl(
            $this->getUrl() . self::ENDPOINT_PAYMENT_GATEWAY
        );

        return $response;
    }

    /**
     * @param SignedApiRequest $apiRequest
     * @return VerifyTransactionResponse
     * @throws ApiResponseException
     */
    public function verifyTransaction(SignedApiRequest $apiRequest): VerifyTransactionResponse
    {
        $apiRequest->setConfig($this->config);

        $response = new VerifyTransactionResponse(
            $this->client()->put(self::ENDPOINT_VERIFY, [
                'json' => $apiRequest->parameters(),
            ])
        );

        return $response;
    }

    /**
     * @throws ApiResponseException
     */
    public function cardChargeTransaction(ParametersApiRequest $apiRequest): CardChargeResponse
    {
        return new CardChargeResponse(
            $this->request(self::ENDPOINT_CARD_CHARGE, $apiRequest->parameters())
        );
    }


    /**
     * @throws ApiResponseException
     */
    public function transactionInfo(string $sessionId): InfoTransactionResponse
    {
        return new InfoTransactionResponse(
            $this->client()->get(self::ENDPOINT_INFO . $sessionId)
        );
    }

    /**
     * @return ClientInterface
     */
    private function client(): ClientInterface
    {

        if (!$this->client) {
            $this->client = new Client([
                'base_uri' => $this->getApiUrl(),
                'auth' => [$this->config->getMerchantId(), $this->config->getApiKey()]
            ]);
        }

        return $this->client;
    }

    /**
     * //
     *
     * @param string $endpoint
     * @param array $parameters
     * @return ResponseInterface
     */
    private function request(string $endpoint, array $parameters): ResponseInterface
    {
        try {
            $response = $this->client()->post($endpoint, [
                'json' => $parameters,
            ]);
        } catch (RequestException $e) {
            $response = $e->getResponse();
        }

        return $response;
    }

    /**
     * @return string
     */
    private function getApiUrl(): string
    {
        $url = $this->config->isLiveMode() ? self::URL_LIVE : self::URL_SANDBOX;

        return $url . self::API_VERSION;
    }

    /**
     * @return string
     */
    private function getUrl(): string
    {
        return $this->config->isLiveMode() ? self::URL_LIVE : self::URL_SANDBOX;
    }
}
