<?php

namespace App\Entity\Subscriptions;

use App\Entity\Orders\Order;
use App\Entity\Subscriptions\Helper\KindOfSubscription;
use App\Entity\Units\Unit;
use App\Entity\Users\User;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Component\Uid\Uuid;

/**
 * @ORM\Entity()
 */
class Subscription
{
    private const MAX_ATTEMPTS = 4;

    /**
     * @ORM\Id
     * @ORM\Column(type="uuid")
     */
    private SubscriptionID|Uuid $uuid;

    /**
     * @ORM\Column(type="integer")
     */
    private int $children;

    /**
     * @ORM\Column(type="integer")
     */
    private int $monthlyCost;

    /**
     * @ORM\Column(type="datetime_immutable")
     */
    private \DateTimeImmutable $createdAt;

    /**
     * @ORM\Column(type="datetime_immutable")
     */
    private \DateTimeImmutable $dateStart;

    /**
     * @ORM\Column(type="datetime_immutable", nullable=true)
     */
    private ?\DateTimeImmutable $dateEnd = null;

    /**
     * @ORM\Column(type="date_immutable", nullable=true)
     */
    private ?\DateTimeImmutable $nextCardCharge;

    /**
     * @ORM\Column(type="datetime_immutable", nullable=true)
     */
    private ?\DateTimeImmutable $dateExpiration = null;

    /**
     * @ORM\Column(type="integer")
     */
    private int $attempts = 0;

    /**
     * @ORM\ManyToOne(targetEntity=User::class)
     * @ORM\JoinColumn(name="user_id", referencedColumnName="uuid", nullable=false)
     */
    private User $user;

    /**
     * @ORM\ManyToOne(targetEntity=Unit::class)
     * @ORM\JoinColumn(name="unit_id", referencedColumnName="uuid")
     */
    private Unit $unit;

    /**
     * @ORM\ManyToOne(targetEntity=Order::class)
     * @ORM\JoinColumn(name="basic_order_id", referencedColumnName="uuid")
     */
    private Order $basicOrderID;

    /**
     * @ORM\Column(type="string", length=1)
     */
    private string $kindOfSubscription;

    /**
     * @ORM\Column(type="boolean", options={"default": 1})
     */
    private bool $isActive = true;

    /**
     * @ORM\Column(type="string", nullable=true)
     */
    private ?string $methodRefID;


    /**
     * @ORM\Column(type="string", nullable=true)
     */
    private ?string $reasonForClosure;

    /**
     * @ORM\ManyToOne(targetEntity=User::class)
     * @ORM\JoinColumn(name="volunteer_id", referencedColumnName="uuid", nullable=true)
     */
    private ?User $volunteer;


    /**
     * @param int $children
     * @param int $monthlyCost
     * @param \DateTimeImmutable $dateStart
     * @param User $user
     * @param Unit $unit
     */
    public function __construct(
        int $children,
        int $monthlyCost,
        \DateTimeImmutable $dateStart,
        User $user,
        Unit $unit,
        Order $basicOrderID
    )
    {
        $this->uuid = SubscriptionID::generate();
        $this->children = $children;
        $this->monthlyCost = $monthlyCost;
        $this->dateStart = $dateStart;
        $this->user = $user;
        $this->unit = $unit;
        $this->basicOrderID = $basicOrderID;
        $this->createdAt = new \DateTimeImmutable();
    }

    public static function monthInterval(): \DateInterval
    {
        return new \DateInterval('P1M');
    }

    public static function paymentErrorInterval(): \DateInterval
    {
        return new \DateInterval('P5D');
    }

    /**
     * @return SubscriptionID
     */
    public function getUuid(): SubscriptionID
    {
        return SubscriptionID::fromString($this->uuid);
    }

    public function deactivate(string $reason)
    {
        $this->isActive = false;
        $this->reasonForClosure = $reason;
        $this->dateEnd = new \DateTimeImmutable();
    }

    public function firstPayment()
    {
        $firstPaymentDateWithMonthlyInterval = $this->dateStart->add(self::monthInterval());

        $this->nextCardCharge = $firstPaymentDateWithMonthlyInterval;
        $this->dateExpiration = $firstPaymentDateWithMonthlyInterval;
    }

    public function cardChargeError()
    {
        $nextChargeDateWithPaymentInterval = $this->nextCardCharge->add(self::paymentErrorInterval());
        $this->nextCardCharge = $nextChargeDateWithPaymentInterval;
        $this->increaseAttempts();
    }

    public function cardChargeSuccess()
    {
        $dateSuccessWithMonthlyInterval = $this->dateExpiration->add(self::monthInterval());

        $this->nextCardCharge = $dateSuccessWithMonthlyInterval;
        $this->dateExpiration = $dateSuccessWithMonthlyInterval;

        $this->resetAttempts();
    }

    public function renew(Order $order)
    {
        $months = $order->getRenewInformation()->getMonths();
        $monthlyCost = $order->getRenewInformation()->getMonthlyCost();

        $this
            ->setDateEndByMonths($months)
            ->setDateExpirationByMonths($months)
            ->setMonthlyCost($monthlyCost)
            ->setBasicOrderID($order);
    }

    public function resetAttempts()
    {
        $this->attempts = 0;
    }

    public function increaseAttempts()
    {
        $this->attempts = $this->attempts + 1;
    }

    /**
     * @return int
     */
    public function getChildren(): int
    {
        return $this->children;
    }

    /**
     * @return int
     */
    public function getMonthlyCost(): int
    {
        return $this->monthlyCost;
    }

    /**
     * @return \DateTimeImmutable
     */
    public function getCreatedAt(): \DateTimeImmutable
    {
        return $this->createdAt;
    }

    /**
     * @return \DateTimeImmutable
     */
    public function getDateStart(): \DateTimeImmutable
    {
        return $this->dateStart;
    }

    /**
     * @return \DateTimeImmutable|null
     */
    public function getDateEnd(): ?\DateTimeImmutable
    {
        return $this->dateEnd;
    }

    /**
     * @return \DateTimeImmutable|null
     */
    public function getDateExpiration(): ?\DateTimeImmutable
    {
        return $this->dateExpiration;
    }

    /**
     * @return User
     */
    public function getUser(): User
    {
        return $this->user;
    }

    /**
     * @return string
     */
    public function getKindOfSubscription(): string
    {
        return $this->kindOfSubscription;
    }

    public function isCyclic(): bool
    {
        return (new KindOfSubscription($this->kindOfSubscription))->isCyclic();
    }

    /**
     * @return string|null
     */
    public function getReasonForClosure(): ?string
    {
        return $this->reasonForClosure;
    }

    /**
     * @return bool
     */
    public function isActive(): bool
    {
        return $this->isActive;
    }

    /**
     * @return string|null
     */
    public function getMethodRefID(): ?string
    {
        return $this->methodRefID;
    }

    /**
     * @param string|null $methodRefID
     * @return Subscription
     */
    public function setMethodRefID(?string $methodRefID): Subscription
    {
        $this->methodRefID = $methodRefID;
        return $this;
    }

    /**
     * @param string $kindOfSubscription
     * @return Subscription
     */
    public function setKindOfSubscription(string $kindOfSubscription): Subscription
    {
        $this->kindOfSubscription = $kindOfSubscription;
        return $this;
    }

    /**
     * @param string|null $reasonForClosure
     * @return Subscription
     */
    public function setReasonForClosure(?string $reasonForClosure): Subscription
    {
        $this->reasonForClosure = $reasonForClosure;
        return $this;
    }

    /**
     * @param int $monthlyCost
     * @return Subscription
     */
    public function setMonthlyCost(int $monthlyCost): Subscription
    {
        $this->monthlyCost = $monthlyCost;
        return $this;
    }

    /**
     * @return Unit
     */
    public function getUnit(): Unit
    {
        return $this->unit;
    }

    /**
     * @return bool
     */
    public function isMaxAttempts(): bool
    {
        return $this->attempts >= self::MAX_ATTEMPTS;
    }


    /**
     * @param int $months
     * @return Subscription
     * @throws \Exception
     */
    public function setDateEndByMonths(int $months): Subscription
    {
        $date = $this->dateEnd ?: $this->dateStart;
        $this->dateEnd = $date->add(new \DateInterval('P' . $months . 'M'));
        return $this;
    }

    /**
     * @param int $months
     * @return Subscription
     * @throws \Exception
     */
    public function setDateExpirationByMonths(int $months): Subscription
    {
        $date = $this->dateExpiration ?: $this->dateStart;

        $this->dateExpiration = $date->add(new \DateInterval('P' . $months . 'M'));
        return $this;
    }

    /**
     * @return int
     */
    public function getAttempts(): int
    {
        return $this->attempts;
    }

    /**
     * @return Order
     */
    public function getBasicOrderID(): Order
    {
        return $this->basicOrderID;
    }

    /**
     * @param Order $basicOrderID
     * @return Subscription
     */
    public function setBasicOrderID(Order $basicOrderID): Subscription
    {
        $this->basicOrderID = $basicOrderID;
        return $this;
    }

    public function getRenewalMonthlyCostByStatus(bool $isUnitMonthlyCost)
    {
        return $isUnitMonthlyCost ? $this->getUnit()?->getHearthAdoption()->getMonthlyCost() : $this->getMonthlyCost();
    }

    /**
     * @return User|null
     */
    public function getVolunteer(): ?User
    {
        return $this->volunteer;
    }

    /**
     * @param User|null $volunteer
     * @return Subscription
     */
    public function setVolunteer(?User $volunteer): Subscription
    {
        $this->volunteer = $volunteer;
        return $this;
    }

    /**
     * @return int
     */
    public function getDaysToEnd(): int
    {
        $dateNow = (new \DateTimeImmutable())->setTime(0, 0);

        $dateEnd = $this->getDateEnd()->setTime(0, 0);

        return $dateNow->diff($dateEnd)->format("%r%a");
    }


}