<?php

namespace App\Entity\BalanceOperations;

use App\Entity\Orders\Order;
use App\Entity\Users\User;
use Doctrine\ORM\Mapping as ORM;
use Doctrine\ORM\Mapping\Entity;
use Symfony\Component\Uid\Uuid;

/**
 * @Entity()
 */
class BalanceOperation
{
    const PERCENTAGE = 0.03;
    /**
     * @ORM\Id
     * @ORM\Column(type="uuid")
     */
    private BalanceOperationID|Uuid $uuid;

    /**
     * @ORM\ManyToOne(targetEntity=User::class)
     * @ORM\JoinColumn(name="user_id", referencedColumnName="uuid", nullable=false)
     */
    private User $user;

    /**
     * @ORM\ManyToOne(targetEntity=Order::class)
     * @ORM\JoinColumn(name="basic_order_id", referencedColumnName="uuid", nullable=true)
     */
    private ?Order $order;

    /**
     * @ORM\Column(type="string", length=255)
     */
    private string $title;

    /**
     * @ORM\Column(type="integer")
     */
    private int $amount;

    /**
     * @ORM\Column(type="integer")
     */
    private int $balance;

    /**
     * @ORM\Column(type="string", length=1)
     */
    private string $type;

    /**
     * @ORM\Column(type="datetime_immutable")
     */
    private \DateTimeImmutable $createdAt;

    /**
     * @param User $user
     * @param string $title
     * @param string $type
     * @param ?Order $order
     */
    public function __construct(
        User $user,
        string $title,
        string $type,
        ?Order $order = null,
    )
    {
        $this->uuid = BalanceOperationID::generate();
        $this->user = $user;
        $this->title = $title;
        $this->type = $type;
        $this->order = $order;
        $this->createdAt = new \DateTimeImmutable();
    }

    /**
     * @return BalanceOperationID
     */
    public function getUuid(): BalanceOperationID
    {
        return BalanceOperationID::fromString($this->uuid);
    }

    /**
     * @return User
     */
    public function getUser(): User
    {
        return $this->user;
    }

    /**
     * @return Order|null
     */
    public function getOrder(): ?Order
    {
        return $this->order;
    }

    /**
     * @return string
     */
    public function getTitle(): string
    {
        return $this->title;
    }

    /**
     * @return int
     */
    public function getAmount(): int
    {
        return $this->amount;
    }

    /**
     * @return int
     */
    public function getBalance(): int
    {
        return $this->balance;
    }

    /**
     * @return string
     */
    public function getType(): string
    {
        return $this->type;
    }

    /**
     * @return \DateTimeImmutable
     */
    public function getCreatedAt(): \DateTimeImmutable
    {
        return $this->createdAt;
    }


    /**
     * @param int $balance
     * @return BalanceOperation
     */
    public function setBalance(int $balance): BalanceOperation
    {
        $this->balance = $balance;
        return $this;
    }

    /**
     * @param int $amount
     * @return BalanceOperation
     */
    public function setAmount(int $amount): BalanceOperation
    {
        $this->amount = $amount;
        return $this;
    }

    public function setAmountByPercentage(int $orderAmount): BalanceOperation
    {
        $this->amount = $orderAmount * self::PERCENTAGE;
        return $this;
    }
}