<?php

namespace App\Core\PdfGenerator;


use App\Core\PdfGenerator\Exceptions\InvalidPaperFormatException;

class PDFBuilder implements PDFBuilderInterface
{
    private const ORIENTATION_LANDSCAPE = 'landscape';
    private const PAPER_FORMATS = ['A1', 'A2', 'A3', 'A4', 'A5'];

    /**
     * @var string
     */
    private $fileName;
    /**
     * @var string
     */
    private $filePath;
    /**
     * @var string
     */
    private $twigName;
    /**
     * @var array
     */
    private $twigParameters = [];
    /**
     * @var string
     */
    private $paperFormat = 'A4';
    /**
     * @var string
     */
    private $orientation = "portrait";

    public static function create(
        string $fileName,
        string $filePath,
        string $twigName
    ): PDFBuilder
    {
        return new PDFBuilder($fileName, $filePath, $twigName);
    }

    private function __construct(
        string $fileName,
        string $filePath,
        string $twigName
    )
    {
        $this->fileName = $fileName;
        $this->filePath = $filePath;
        $this->twigName = $twigName;
    }

    public function setTwigParameters(PDFParametersBuilderInterface $PDFParametersBuilder): self
    {
        $this->twigParameters = $PDFParametersBuilder->getParameters();
        return $this;
    }

    public function changeOrientationToLandscape(): self
    {
        $this->orientation = self::ORIENTATION_LANDSCAPE;
        return $this;
    }

    /**
     * @throws InvalidPaperFormatException
     */
    public function changePaperFormat(string $paperFormat): self
    {
        if (!in_array($paperFormat, self::PAPER_FORMATS)) {
            throw new InvalidPaperFormatException();
        }

        $this->paperFormat = $paperFormat;

        return $this;
    }

    public function build(): PDFFile
    {
        return new PDFFile(
            $this->fileName,
            $this->filePath,
            $this->twigName,
            $this->twigParameters,
            $this->paperFormat,
            $this->orientation
        );
    }
}