<?php

namespace App\Core\EventSubscriber;

use App\Core\Exception\ApiException;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\RateLimiter\RateLimiterFactory;
use Symfony\Component\Security\Http\Authenticator\Passport\Badge\UserBadge;
use Symfony\Component\Security\Http\Authenticator\Passport\PassportInterface;
use Symfony\Component\Security\Http\Event\CheckPassportEvent;
use Symfony\Component\Security\Http\Event\LoginSuccessEvent;
use Symfony\Contracts\Translation\TranslatorInterface;

final class RateLimiterSubscriber implements EventSubscriberInterface
{
    public function __construct(
        private RequestStack $requestStack,
        private RateLimiterFactory $customLoginThrottlingLimiter,
        private TranslatorInterface $translator
    ) {}

    public static function getSubscribedEvents(): array
    {
        return [
            CheckPassportEvent::class => ['checkPassport', 2080],
            LoginSuccessEvent::class => 'onSuccessfulLogin'
        ];
    }

    /**
     * @throws ApiException
     */
    public function checkPassport(CheckPassportEvent $event): void
    {
        $passport = $event->getPassport();
        if (!$passport->hasBadge(UserBadge::class)) {
            return;
        }

        $request = $this->requestStack->getMainRequest();

        $key = $this->generateUserAuthKey($passport, $request);
        $limiter = $this->customLoginThrottlingLimiter->create($key);

        if (!$limiter->consume()->isAccepted()) {
            throw new ApiException(
                $this->translator->trans('login.attempts_exceeded'),
                Response::HTTP_CONFLICT
            );
        }
    }

    public function onSuccessfulLogin(LoginSuccessEvent $event): void
    {
        $key = $this->generateUserAuthKey($event->getPassport(), $event->getRequest());
        $this->customLoginThrottlingLimiter->create($key)->reset();
    }

    private function generateUserAuthKey(PassportInterface $passport, Request $request): string
    {
        return $passport->getBadge(UserBadge::class)->getUserIdentifier() . '-' . $request->getClientIp();
    }
}