<?php

namespace App\Controller\Users\Update;

use App\Entity\Permissions\Permission;
use App\Entity\Roles\Role;
use App\Entity\Users\User;
use App\Models\Users\UserModel;
use App\Services\Users\UpdateUserPersonalData;
use App\Services\Users\UpdateUserRole;
use Nelmio\ApiDocBundle\Annotation\Model;
use OpenApi\Annotations as OA;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\IsGranted;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\ParamConverter;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\Routing\Annotation\Route;

/**
 * @OA\Tag(name="Users")
 */
#[Route('/api/users', name: 'users_')]
class UpdateUserController extends AbstractController
{
    public function __construct(
        private UpdateUserPersonalData $updateUserPersonalData,
        private UpdateUserRole $userRole
    ) {}

    /**
     * @OA\Put(summary="Update user personal data")
     * @OA\RequestBody(
     *     @Model(type=UserModel::class, groups=UserModel::PERSONAL_DATA)
     * )
     */
    #[IsGranted(Permission::ACCOUNT_DATA_MANAGEMENT)]
    #[IsGranted(Permission::PRIVATE_ACCESS, 'user')]
    #[Route(path: '/{userId}/personal-data', name: 'edit_personal_data', methods: ['PUT'])]
    #[ParamConverter('user', options: ['mapping' => ['userId' => 'uuid']])]
    #[ParamConverter('userModel', options: ["groups" => [...UserModel::PERSONAL_DATA]], converter: "json_deserialize_param_converter")]
    public function editUserData(User $user, UserModel $userModel): JsonResponse
    {
        $this->updateUserPersonalData->update($user, $userModel);
        return $this->json(['User personal data has been updated.']);
    }

    /**
     * @OA\Put(summary="Update user role")
     */
    #[IsGranted(Permission::USERS_ADMINISTRATION)]
    #[Route(path: '/{userId}/changeRole/{roleId}', name: 'edit_password', methods: ['PUT'])]
    #[ParamConverter('user', options: ['mapping' => ['userId' => 'uuid']])]
    #[ParamConverter('role', options: ['mapping' => ['roleId' => 'uuid']])]
    public function editUserRole(User $user, Role $role): JsonResponse
    {
        $this->userRole->editUserRole($user, $role);
        return $this->json(['User role has been updated.']);
    }
}