<?php

namespace App\Controller\Users\Data;

use App\Core\Paginator\ObjectValue\PaginationRequest;
use App\Entity\Permissions\Permission;
use App\Entity\Users\ReadModel\UsersQueryInterface;
use App\Entity\Users\User;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\IsGranted;
use App\Core\Paginator\ObjectValue\PaginationResults;
use App\Entity\Users\UserID;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\ParamConverter;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Annotation\Route;
use OpenApi\Annotations as OA;
use Symfony\Component\Serializer\Normalizer\AbstractNormalizer;

/** @OA\Tag(name="Users") */
#[Route('/api/users', name: 'users_')]
class UserDataController extends AbstractController
{
    public function __construct(
        private UsersQueryInterface $usersQuery,
        private PaginationResults $paginationResults
    ) {}

    /**
     * @OA\Get(summary="User list")
     * @OA\Parameter(name="page", in="query", description="page number")
     * @OA\Parameter(name="perPage", in="query", description="item per page")
     */
    #[IsGranted(Permission::USERS_ADMINISTRATION)]
    #[Route(name: 'list', methods: ['GET'])]
    public function userList(Request $request): JsonResponse
    {
        $results = $this->usersQuery->findAllUsers(PaginationRequest::build($request));
        return $this->json(
            $this->paginationResults->data($results)->totalRow()
        );
    }

    /** @OA\Get(summary="Show user personal data") */
    #[IsGranted(Permission::ACCOUNT_DATA_MANAGEMENT)]
    #[IsGranted(Permission::PRIVATE_ACCESS, 'user')]
    #[Route(path: '/{userId}/personal-data', name: 'show_personal_data', methods: ['GET'])]
    #[ParamConverter('user', options: ['mapping' => ['userId' => 'uuid']])]
    public function showUserPersonalData(User $user): JsonResponse
    {
        return $this->json(
           $this->usersQuery->findOneByUserID($user->getUuid()), context: [AbstractNormalizer::IGNORED_ATTRIBUTES => ['permissions']]
        );
    }
}