<?php

namespace App\Controller\Units\Data;

use App\Core\Paginator\ObjectValue\PaginationRequest;
use App\Entity\Permissions\Permission;
use App\Core\Paginator\ObjectValue\PaginationResults;
use App\Entity\Units\ReadModel\UnitsQueryInterface;
use App\Entity\Units\UnitID;
use App\Entity\Users\User;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\IsGranted;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Annotation\Route;
use OpenApi\Annotations as OA;

/**
 * @OA\Tag(name="Units")
 */
#[Route(path: '/api/units', name: 'units_')]
class UnitDataController extends AbstractController
{
    public function __construct(
        private UnitsQueryInterface $unitsQuery,
        private PaginationResults $paginationResults
    ) {}

    /**
     * @OA\Get(summary="Active unit list")
     * @OA\Parameter(name="page", in="query", description="page number")
     * @OA\Parameter(name="perPage", in="query", description="item per page")
     */
    #[IsGranted(Permission::PUBLIC_ACCESS)]
    #[Route(path: '/active', name: 'active_list', methods: ['GET'])]
    public function activeUnitsList(Request $request): JsonResponse
    {
        $results = $this->unitsQuery->findAllActiveUnits(PaginationRequest::build($request));
        return $this->json(
            $this->paginationResults->data($results)->totalRow()
        );
    }

    /**
     * @OA\Get(summary="Unit list")
     * @OA\Parameter(name="page", in="query", description="page number")
     * @OA\Parameter(name="perPage", in="query", description="item per page")
     */
    #[IsGranted(Permission::UNIT_ADMINISTRATION)]
    #[Route(name: 'list', methods: ['GET'])]
    public function unitsList(Request $request): JsonResponse
    {
        $results = $this->unitsQuery->findAllUnits(PaginationRequest::build($request));
        return $this->json(
            $this->paginationResults->data($results)->totalRow()
        );
    }

    /**
     * @OA\Get(summary="Show unit data")
     */
    #[IsGranted(Permission::PUBLIC_ACCESS)]
    #[Route(path: '/{unitId}', name: 'show_data', methods: ['GET'])]
    public function showUnitData(string $unitId): JsonResponse
    {
        /** @var User $user */
        $user = $this->getUser();
        $userUUid = $user?->getUuid();
        return $this->json(
            $this->unitsQuery->findUnitDataById(UnitID::fromString($unitId), $userUUid)
        );
    }

    /**
     * @OA\Get(summary="Show unit details needed for editing")
     */
    #[IsGranted(Permission::UNIT_ADMINISTRATION)]
    #[Route(path: '/{unitId}/details', name: 'show_details', methods: ['GET'])]
    public function showUnitDetails(string $unitId): JsonResponse
    {
        return $this->json(
            $this->unitsQuery->findUnitDetailsByUnitId(UnitID::fromString($unitId))
        );
    }
}