<?php

namespace App\Controller\Units\Create;

use App\Entity\Permissions\Permission;
use App\Models\Units\UnitModel;
use App\Services\Units\CreateUnit;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\IsGranted;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\ParamConverter;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\Routing\Annotation\Route;
use OpenApi\Annotations as OA;

/**
 * @OA\Tag(name="Units")
 */
#[Route(path: '/api/units', name: 'units_')]
class CreateUnitController extends AbstractController
{
    public function __construct(
        private CreateUnit $createUnit
    ) {}

    /**
     * @OA\Post(summary="Create new unit")
     * @OA\RequestBody(
     *     @OA\MediaType(
     *          mediaType="multipart/form-data",
     *          @OA\Schema(
     *              type="object",
     *              required={"unitType", "title", "description", "photo", "banner"},
     *              @OA\Property(property="unitType", type="string"),
     *              @OA\Property(property="title", type="string"),
     *              @OA\Property(property="description", type="string"),
     *              @OA\Property(property="photo", type="file"),
     *              @OA\Property(property="banner", type="file"),
     *              @OA\Property(property="unitChildren", type="string")
     *          )
     *     )
     * )
     * @OA\Response(response=200, description="Returns the uuid of the created unit.")
     */
    #[IsGranted(Permission::UNIT_ADMINISTRATION)]
    #[Route(name: 'create', methods: ['POST'])]
    #[ParamConverter('unitModel', options: ["groups" => [UnitModel::CREATE]], converter: 'form_data_deserialize_param_converter')]
    public function create(UnitModel $unitModel): JsonResponse
    {
        $unitId = $this->createUnit->create($unitModel);
        return $this->json(['unitId' => $unitId]);
    }
}