<?php

namespace App\Controller\Reports;

use App\Core\ValidatorModel\ValidatorModelInterface;
use App\Entity\Permissions\Permission;
use App\Entity\Users\User;
use App\Models\Reports\RequestDatesModel;
use App\Services\Reports\GenerateAnnualReportPDF;
use OpenApi\Annotations as OA;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\IsGranted;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\ParamConverter;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Symfony\Component\HttpFoundation\File\Stream;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Annotation\Route;

/** @OA\Tag(name="Reports") */
#[Route('/api/reports', name: 'reports_')]
class AnnualReportController extends AbstractController
{
    public function __construct(
        private ValidatorModelInterface $validator,
        private GenerateAnnualReportPDF $generateAnnualReportPDF
    )
    {
    }

    /**
     * @OA\Get(summary="Get finances report data")
     * @OA\Parameter(name="year", in="query", description="year start")
     * @param Request $request
     * @param User $user
     * @return BinaryFileResponse
     */
    #[IsGranted(Permission::ACCOUNT_DATA_MANAGEMENT)]
    #[IsGranted(Permission::PRIVATE_ACCESS, 'user')]
    #[Route(path: '/{userId}/annual', name: 'annual', methods: ['GET'])]
    #[ParamConverter('user', options: ['mapping' => ['userId' => 'uuid']])]
    public function yearReport(Request $request, User $user): BinaryFileResponse
    {
        $year = $request->query->get('year');

        $requestDates = new RequestDatesModel(year: $year);
        $this->validator->validate($requestDates);

        $PDFFile = $this->generateAnnualReportPDF->generate($user, $requestDates);

        $stream = new Stream($PDFFile->getFilePath() . $PDFFile->getFileName());
        $response = new BinaryFileResponse($stream);
        $response->setContentDisposition('inline', $stream->getBasename());
        $response->headers->set('X-suggested-filename', $stream->getBasename());
        $response->headers->set('Content-type', 'application/pdf');

        return $response;
    }


    #[IsGranted(Permission::UNIT_ADMINISTRATION)]
    #[Route(path: '/annual/web', name: 'annual_web', methods: ['GET'])]
    public function yearWeb()
    {
        return $this->render('pdf/annual_report.html.twig');
    }
}