<?php

namespace App\Controller\Posts;

use App\Entity\Permissions\Permission;
use App\Entity\Posts\Post;
use App\Models\Posts\PostModel;
use App\Services\Posts\UpdatePost;
use App\Services\Posts\UpdatePostStatus;
use OpenApi\Annotations as OA;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\ParamConverter;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Security;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\Routing\Annotation\Route;

/**
 * @OA\Tag(name="Posts")
 */
#[Route(path: '/api/posts', name: 'posts_')]
class UpdatePostController extends AbstractController
{
    public function __construct(
        private UpdatePost $updatePost,
        private UpdatePostStatus $postStatus
    ) {}

    /**
     * @OA\Post(summary="Update post")
     * @OA\RequestBody(
     *     @OA\MediaType(
     *          mediaType="multipart/form-data",
     *          @OA\Schema(
     *              type="object",
     *              required={"title", "description", "content"},
     *              @OA\Property(property="title", type="string"),
     *              @OA\Property(property="description", type="string"),
     *              @OA\Property(property="content", type="string"),
     *              @OA\Property(property="photo", type="file")
     *          )
     *     )
     * )
     */
    #[Security("is_granted('ROLE_UNIT_ADMINISTRATION') or is_granted('ROLE_UNIT_REDACTION')")]
    #[Route(path: '/{postId}', name: 'update', methods: ['POST'])]
    #[ParamConverter('post', options: ['mapping' => ['postId' => 'uuid']])]
    public function update(Post $post, PostModel $postModel): JsonResponse
    {
        $this->updatePost->update($post, $postModel);
        return $this->json('Post has been successful updated.');
    }

    #[Security("is_granted('ROLE_UNIT_ADMINISTRATION') or is_granted('ROLE_UNIT_REDACTION')")]
    #[Route(path: '/{postId}/publish', name: 'publish', methods: ['PUT'])]
    #[ParamConverter('post', options: ['mapping' => ['postId' => 'uuid']])]
    public function publish(Post $post): JsonResponse
    {
        $this->postStatus->publish($post);
        return $this->json('Post has been successful published');
    }

    #[Security("is_granted('ROLE_UNIT_ADMINISTRATION') or is_granted('ROLE_UNIT_REDACTION')")]
    #[Route(path: '/{postId}/unpublish', name: 'unpublish', methods: ['PUT'])]
    #[ParamConverter('post', options: ['mapping' => ['postId' => 'uuid']])]
    public function unpublish(Post $post): JsonResponse
    {
        $this->postStatus->unpublish($post);
        return $this->json('Post has been successful unpublished');
    }
}