<?php

namespace App\Entity\Users;

use App\Core\Users\TokenGenerator;
use App\Entity\Agreement\Agreement;
use App\Entity\Permissions\Permission;
use App\Entity\Units\Unit;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use App\Entity\Roles\Role;
use Doctrine\ORM\Mapping as ORM;
use PhpParser\Node\Expr\Closure;
use Symfony\Component\Notifier\Recipient\EmailRecipientInterface;
use Symfony\Component\Security\Core\User\UserInterface;
use Symfony\Component\Uid\Uuid;

/**
 * @ORM\Entity()
 */
class User implements UserInterface, \JsonSerializable, EmailRecipientInterface
{
    /**
     * @ORM\Id
     * @ORM\Column(type="uuid")
     */
    private UserID|Uuid $uuid;

    /**
     * @ORM\Column(type="string", length=50)
     */
    private string $name;

    /**
     * @ORM\Column(type="string", length=50)
     */
    private string $surname;

    /**
     * @ORM\Column(type="string", length=180, unique=true)
     */
    private string $email;

    /**
     * @ORM\ManyToOne(targetEntity=Role::class)
     * @ORM\JoinColumn(name="role_id", referencedColumnName="uuid", nullable=false)
     */
    private Role $role;

    /**
     * @var string The hashed password
     * @ORM\Column(type="string")
     */
    private string $password;

    /**
     * @ORM\Column(type="boolean", options={"default": 0})
     */
    private bool $isActive = false;

    /**
     * @ORM\Column(type="boolean", options={"default": 0})
     */
    private bool $isBlocked = false;

    /**
     * @ORM\Column(type="string", length=255, nullable=true, unique=true)
     */
    private ?string $token;

    /**
     * @ORM\Column(type="string", length=4, nullable=true, unique=true)
     */
    private ?string $referenceCode = null;

    /**
     * @ORM\ManyToMany(targetEntity=Agreement::class)
     * @ORM\JoinTable(
     *     name="user_agreement",
     *     joinColumns={@ORM\JoinColumn(name="user_id", referencedColumnName="uuid")},
     *     inverseJoinColumns={@ORM\JoinColumn(name="agreement_id", referencedColumnName="uuid")}
     * )
     */
    private Collection $agreements;

    /**
     * @ORM\ManyToMany(targetEntity=Unit::class)
     * @ORM\JoinTable(
     *     name="redactor_partners",
     *     joinColumns={@ORM\JoinColumn(name="user_id", referencedColumnName="uuid")},
     *     inverseJoinColumns={@ORM\JoinColumn(name="unit_id", referencedColumnName="uuid")}
     * )
     */
    private Collection $assignedUnits;

    public function __construct(UserID $uuid, string $name, string $surname, string $email, Role $role)
    {
        $this->uuid = $uuid;
        $this->name = $name;
        $this->surname = $surname;
        $this->email = $email;
        $this->role = $role;
        $this->agreements = new ArrayCollection();
        $this->assignedUnits = new ArrayCollection();
    }

    /**
     * @return UserID
     */
    public function getUuid(): UserID
    {
        return UserID::fromString($this->uuid);
    }

    public function getUserIdentifier(): string
    {
        return $this->email;
    }

    /**
     * @return string
     */
    public function getName(): string
    {
        return $this->name;
    }

    /**
     * @param string $name
     */
    public function setName(string $name): void
    {
        $this->name = $name;
    }

    /**
     * @return string
     */
    public function getSurname(): string
    {
        return $this->surname;
    }

    /**
     * @param string $surname
     */
    public function setSurname(string $surname): void
    {
        $this->surname = $surname;
    }

    /**
     * @param string $name
     * @param string $surname
     */
    public function setPersonalData(string $name, string $surname): void
    {
        $this->name = $name;
        $this->surname = $surname;
    }

    public function getEmail(): string
    {
        return $this->email;
    }

    /**
     * A visual identifier that represents this user.
     *
     * @see UserInterface
     */
    public function getUsername(): string
    {
        return (string)$this->email;
    }

    /**
     * @return array
     */
    public function getRoles(): array
    {
        $data = array_map(
            fn($permission) => $permission->getAuthorization(),
            $this->role->getPermissions()
        );
//        $data[] = Permission::PRIVATE_ACCESS;
        return $data;
    }

    public function updateRole(Role $role): void
    {
        $this->role = $role;
    }

    /**
     * @see UserInterface
     */
    public function getPassword(): string
    {
        return (string)$this->password;
    }

    /**
     * @param string $password
     */
    public function setPassword(string $password): void
    {
        $this->password = password_hash($password, PASSWORD_ARGON2I);
    }

    /**
     * Returning a salt is only needed, if you are not using a modern
     * hashing algorithm (e.g. bcrypt or sodium) in your security.yaml.
     *
     * @see UserInterface
     */
    public function getSalt(): ?string
    {
        return null;
    }

    /**
     * @see UserInterface
     */
    public function eraseCredentials()
    {
        // If you store any temporary, sensitive data on the user, clear it here
        // $this->plainPassword = null;
    }

    public function getIsActive(): ?bool
    {
        return $this->isActive;
    }

    public function getIsNotActive(): ?bool
    {
        return !$this->isActive;
    }

    public function activate(): void
    {
        $this->clearToken();
        $this->isActive = true;
    }

    /**
     * @return bool
     */
    public function getIsBlocked(): bool
    {
        return $this->isBlocked;
    }

    /**
     * @return bool
     */
    public function isBlocked(): bool
    {
        return $this->isBlocked;
    }

    /**
     * @return string|null
     */
    public function getReferenceCode(): ?string
    {
        return $this->referenceCode;
    }

    /**
     * @param string|null $referenceCode
     * @return User
     */
    public function setReferenceCode(?string $referenceCode): User
    {
        $this->referenceCode = $referenceCode;
        return $this;
    }

    /**
     * @param bool $isBlocked
     */
    public function setIsBlocked(bool $isBlocked): void
    {
        $this->isBlocked = $isBlocked;
    }

    public function block()
    {
        $this->isBlocked = true;
    }

    public function unblock()
    {
        $this->isBlocked = false;
    }

    public function generateToken(): void
    {
        $tokenGenerator = new TokenGenerator();
        $this->token = $tokenGenerator->generateToken();
    }

    public function getToken(): ?string
    {
        return $this->token;
    }

    public function clearToken(): void
    {
        $this->token = null;
    }

    public function getAgreementsIDs(): array
    {
        return $this->agreements
            ->map(fn($agreement) => $agreement->getUuid())
            ->toArray();
    }

    /**
     * @return Collection|Agreement[]
     */
    public function getAgreements(): Collection
    {
        return $this->agreements;
    }

    public function addAgreement(Agreement $agreement): self
    {
        if (!$this->agreements->contains($agreement)) {
            $this->agreements[] = $agreement;
        }

        return $this;
    }

    public function removeAgreement(Agreement $agreement): self
    {
        $this->agreements->removeElement($agreement);

        return $this;
    }

    public function hasSubscriptionExpiringAgreement(): bool
    {
        $collection = $this->agreements->filter(function (Agreement $agreement) {
            return $agreement->getOriginalName() === Agreement::ONE_TIME_SUBSCRIPTION_EXPIRING;
        });

        return !$collection->isEmpty();
    }

    /**
     * @return Collection|Unit[]
     */
    public function getAssignedUnits(): Collection
    {
        return $this->assignedUnits;
    }

    public function getAssignedUnitsIDs(): array
    {
        return $this->assignedUnits
            ->map(fn($unit) => $unit->getUuid())
            ->toArray();
    }

    public function addAssignedUnit(Unit $unit): self
    {
        if (!$this->assignedUnits->contains($unit)) {
            $this->assignedUnits[] = $unit;
        }
        return $this;
    }

    public function removeAssignedUnit(Unit $unit): self
    {
        $this->assignedUnits->removeElement($unit);

        return $this;
    }

    public function jsonSerialize()
    {
        return [
            'uuid' => $this->getUuid(),
            'email' => $this->getEmail(),
            'roles' => $this->getRoles()
        ];
    }

    public function hasUnitAdministrationPermission(): bool
    {
        return $this->role->hasUnitAdministrationPermission();
    }

    public function hasUnitRedactionPermission(): bool
    {
        return $this->role->hasUnitRedactionPermission();
    }
}
