<?php

namespace App\Entity\Roles;

use App\Entity\Permissions\Permission;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Component\Uid\Uuid;

/**
 * @ORM\Entity()
 */
class Role
{
    const ADMIN = 'ADMIN';
    const EDITOR = 'EDITOR';
    const USER = 'USER';
    const VOLUNTEER = 'VOLUNTEER';

    const REDACTOR_PARTNER = 'REDACTOR_PARTNER';

    /**
     * @ORM\Id
     * @ORM\Column(type="uuid")
     */
    private RoleID|Uuid $uuid;

    /**
     * @ORM\Column(type="string", unique=true)
     */
    private string $name;

    /**
     * @ORM\Column(type="string", unique=true, nullable=true)
     */
    private ?string $originalName = null;

    /**
     * @ORM\ManyToMany(targetEntity=Permission::class)
     * @ORM\JoinTable(
     *     name="role_permission",
     *     joinColumns={@ORM\JoinColumn(name="role_id", referencedColumnName="uuid")},
     *     inverseJoinColumns={@ORM\JoinColumn(name="permission_id", referencedColumnName="uuid")}
     * )
     */
    private Collection $permissions;

    /**
     * @param string $name
     */
    public function __construct(string $name)
    {
        $this->uuid = RoleID::generate();
        $this->name = $name;
        $this->permissions = new ArrayCollection();
    }

    /**
     * @return RoleID
     */
    public function getUuid(): RoleID
    {
        return RoleID::fromString($this->uuid);
    }

    /**
     * @return string
     */
    public function getName(): string
    {
        return $this->name;
    }

    /**
     * @return string|null
     */
    public function getOriginalName(): ?string
    {
        return $this->originalName;
    }

    /**
     * @param string|null $originalName
     */
    public function setOriginalName(?string $originalName): void
    {
        $this->originalName = $originalName;
    }

    /**
     * @return Permission[]
     */
    public function getPermissions(): array
    {
        return $this->permissions->toArray();
    }

    public function addPermission(Permission $permission): void
    {
        if (!$this->permissions->contains($permission)) {
            $this->permissions[] = $permission;
        }
    }

    public function removePermission(Permission $permission): void
    {
        $this->permissions->removeElement($permission);
    }

    public function isVolunteer(): bool
    {
        return $this->originalName === self::VOLUNTEER;
    }

    public function hasUnitAdministrationPermission(): bool
    {
        $collection = $this->permissions->filter(function (Permission $permission) {
            return $permission->getAuthorization() === Permission::UNIT_ADMINISTRATION;
        });

        return !$collection->isEmpty();
    }

    public function hasUnitRedactionPermission(): bool
    {
        $collection = $this->permissions->filter(function (Permission $permission) {
            return $permission->getAuthorization() === Permission::UNIT_REDACTION;
        });

        return !$collection->isEmpty();
    }
}