<?php

namespace App\Entity\Reports\ReadModel;

use App\Core\Utils\FromArray;
use App\Core\Utils\Money;
use App\Entity\Orders\OrderID;
use App\Entity\Orders\ReadModel\OrderActionTranslations;
use App\Entity\Units\UnitID;
use App\Entity\Users\UserID;
use App\ReportSystem\OrderReportConfig;

class OrderReportDTO
{
    use FromArray;

    public function __construct(
        private string $orderId,
        private string $unitId,
        private string $unitTitle,
        private string $email,
        private string $status,
        private string $verifiedAt,
        private int $amount,
        private bool $hasCommercialAgreement,
        private ?string $userId = null,
        private ?string $userName = null,
        private ?string $userSurname = null,
        private ?string $referenceCode = null,
        private ?int $months = null,
        private ?string $transactionNumber = null
    )
    {
        $this->orderId = (string)OrderID::fromString($this->orderId);
        $this->unitId = (string)UnitID::fromString($this->unitId);
        $this->userId = $this->userId ? (string)UserID::fromString($this->userId) : null;
    }

    /**
     * @return string
     */
    public function getOrderId(): string
    {
        return $this->orderId;
    }

    /**
     * @return string
     */
    public function getUnitId(): string
    {
        return $this->unitId;
    }

    /**
     * @return string
     */
    public function getUnitTitle(): string
    {
        return $this->unitTitle;
    }

    /**
     * @return string|null
     */
    public function getUserId(): ?string
    {
        return $this->userId;
    }

    /**
     * @return string|null
     */
    public function getUserName(): ?string
    {
        return $this->userName;
    }

    /**
     * @return string|null
     */
    public function getUserSurname(): ?string
    {
        return $this->userSurname;
    }

    /**
     * @return string
     */
    public function getEmail(): string
    {
        return $this->email;
    }

    /**
     * @return string|null
     */
    public function getReferenceCode(): ?string
    {
        return $this->referenceCode;
    }

    /**
     * @return string
     */
    public function getStatus(): string
    {
        return (string)new OrderActionTranslations($this->status);
    }

    /**
     * @return string
     * @throws \Exception
     */
    public function getDate(): string
    {
        return (new \DateTimeImmutable($this->verifiedAt))->format('Y-m-d');
    }

    /**
     * @return string
     * @throws \Exception
     */
    public function getHours(): string
    {
        return (new \DateTimeImmutable($this->verifiedAt))->format('H:i');
    }

    /**
     * @return float
     */
    public function getMoney(): float
    {
        return Money::toRational($this->amount);
    }

    /**
     * @return string
     */
    public function getHasCommercialAgreement(): string
    {
        return $this->hasCommercialAgreement ? OrderReportConfig::COMMERCIAL_AGREEMENT_EXIST : OrderReportConfig::COMMERCIAL_AGREEMENT_NOT_EXIST;
    }

    /**
     * @return int|null
     */
    public function getMonths(): ?int
    {
        return $this->months;
    }

    /**
     * @return string|null
     */
    public function getTransactionNumber(): ?string
    {
        return $this->transactionNumber;
    }


}