<?php

namespace App\Entity\Posts;

use App\Entity\Contents\Content;
use App\Entity\Images\Image;
use App\Entity\Units\Unit;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Component\Uid\Uuid;

/** @ORM\Entity() */
class Post
{
    /**
     * @ORM\Id
     * @ORM\Column(type="uuid")
     */
    private PostID|Uuid $uuid;

    /**
     * @ORM\ManyToOne(targetEntity=Unit::class)
     * @ORM\JoinColumn(name="unit_id", referencedColumnName="uuid", nullable=false)
     */
    private Unit $unit;

    /**
     * @ORM\Column(type="string", length=100)
     */
    private string $title;

    /**
     * @ORM\Column(type="string", length=200)
     */
    private string $description;

    /**
     * @ORM\OneToOne(targetEntity=Image::class, cascade={"persist", "remove"})
     * @ORM\JoinColumn(name="photo_id", referencedColumnName="uuid", nullable=true)
     */
    private ?Image $photo = null;

    /**
     * @ORM\OneToOne(targetEntity=Content::class, cascade={"persist", "remove"})
     * @ORM\JoinColumn(name="content_id", referencedColumnName="uuid", nullable=false)
     */
    private Content $postContent;

    /**
     * @ORM\Column(type="datetime_immutable")
     */
    private \DateTimeImmutable $createdAt;

    /**
     * @ORM\Column(type="boolean", options={"default": 1})
     */
    private bool $isActive = true;

    /**
     * @ORM\Column(type="boolean", options={"default": 0})
     */
    private bool $isPublished = false;

    public function __construct(PostID $postID, Unit $unit, string $title, string $description, Content $postContent)
    {
        $this->uuid = $postID;
        $this->unit = $unit;
        $this->title = $title;
        $this->description = $description;
        $this->postContent = $postContent;
        $this->createdAt = new \DateTimeImmutable();
    }

    /** @return PostID */
    public function getUuid(): PostID
    {
        return PostID::fromString($this->uuid);
    }

    /** @return Unit */
    public function getUnit(): Unit
    {
        return $this->unit;
    }

    /**
     * @return string
     */
    public function getTitle(): string
    {
        return $this->title;
    }

    /**
     * @param string $title
     */
    public function setTitle(string $title): void
    {
        $this->title = $title;
    }

    /**
     * @return string
     */
    public function getDescription(): string
    {
        return $this->description;
    }

    /**
     * @param string $description
     */
    public function setDescription(string $description): void
    {
        $this->description = $description;
    }

    /**
     * @return Image|null
     */
    public function getPhoto(): ?Image
    {
        return $this->photo;
    }

    /**
     * @param Image|null $photo
     */
    public function setPhoto(?Image $photo): void
    {
        $this->photo = $photo;
    }

    /**
     * @param Image|null $photo
     */
    public function updatePhoto(?Image $photo): void
    {
        if ($photo) {
            $this->photo = $photo;
        }
    }

    /**
     * @return Content
     */
    public function getPostContent(): Content
    {
        return $this->postContent;
    }

    /**
     * @return \DateTimeImmutable
     */
    public function getCreatedAt(): \DateTimeImmutable
    {
        return $this->createdAt;
    }

    public function updateData(
        string $title,
        string $description,
        string $content
    ): void
    {
        $this->title = $title;
        $this->description = $description;
        $this->postContent->updateContent($content);
    }

    /**
     * @return bool
     */
    public function isNotActive(): bool
    {
        return !$this->isActive;
    }

    public function deactivate(): void
    {
        $this->isActive = false;
    }

    /**
     * @return bool
     */
    public function isPublished(): bool
    {
        return $this->isPublished;
    }

    /**
     * @return void
     */
    public function publish(): void
    {
        $this->isPublished = true;
    }

    /**
     * @return void
     */
    public function unpublish(): void
    {
        $this->isPublished = false;
    }

}