<?php

namespace App\Entity\Orders;

use App\Entity\BalanceOperations\Helper\Type;
use App\Entity\Orders\Helper\OrderAction;
use App\Entity\Orders\Helper\OrderStatus;
use App\Entity\Orders\Helper\RenewInformation;
use App\Entity\Payments\Payment;
use App\Entity\Subscriptions\Subscription;
use App\Entity\Units\Unit;
use App\Entity\Users\User;
use DateTimeImmutable;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Component\Uid\Uuid;

/**
 * @ORM\Entity()
 * @ORM\Table("sysOrder")
 */
class Order implements \JsonSerializable
{
    /**
     * @ORM\Id
     * @ORM\Column(type="uuid")
     */
    private OrderID|Uuid $uuid;

    /**
     * @ORM\ManyToOne(targetEntity=User::class)
     * @ORM\JoinColumn(name="user_id", referencedColumnName="uuid", nullable=true)
     */
    private ?User $user;

    /**
     * @ORM\ManyToOne(targetEntity=Unit::class)
     * @ORM\JoinColumn(name="unit_id", referencedColumnName="uuid")
     */
    private Unit $unit;

    /**
     * @ORM\ManyToOne(targetEntity=Subscription::class, cascade={"persist"})
     * @ORM\JoinColumn(name="subscription_id", referencedColumnName="uuid", nullable=true)
     */
    private ?Subscription $subscription;

    /**
     * @ORM\Column(type="string", length=255)
     */
    private string $email;

    /**
     * @ORM\Column(type="string", length=255)
     */
    private string $description;

    /**
     * @ORM\Column(type="string", length=255)
     */
    private string $status;

    /**
     * @ORM\Column(type="string", length=255)
     */
    private string $token;

    /**
     * @ORM\Column(type="datetime_immutable")
     */
    private DateTimeImmutable $createdAt;

    /**
     * @ORM\Column(type="datetime_immutable", nullable=true)
     */
    private ?DateTimeImmutable $verifiedAt;

    /**
     * @ORM\Column(type="integer",  nullable=true)
     */
    private ?int $externalID;

    /**
     * @ORM\Column(type="integer")
     */
    private int $amount;

    /**
     * @ORM\Column(type="integer", nullable=true)
     */
    private ?int $months = null;

    /**
     * @ORM\Column(type="string", length=1)
     */
    private string $action;

    /**
     * @ORM\Column(type="string", length=255)
     */
    private string $urlReturn;

    /**
     * @ORM\Column(type="string", length=255)
     */
    private string $urlWebhook;

    private string $urlCardInfo;

    /**
     * @ORM\Embedded(class="App\Entity\Orders\Helper\RenewInformation")
     */
    private RenewInformation $renewInformation;

    /**
     * @ORM\OneToMany(targetEntity="App\Entity\Payments\Payment", mappedBy="order", cascade={"persist"})
     */
    private Collection $payments;

    /**
     * @ORM\ManyToOne(targetEntity=User::class)
     * @ORM\JoinColumn(name="volunteer_id", referencedColumnName="uuid", nullable=true)
     */
    private ?User $volunteer;

    /**
     * @ORM\Column(type="string", length=15)
     */
    private string $transactionNumber = '';


    public function __construct(Unit $unit, int $amount)
    {
        $this->uuid = OrderID::generate();
        $this->unit = $unit;
        $this->amount = $amount;
        $this->payments = new ArrayCollection();
        $this->createdAt = new DateTimeImmutable();
        $this->renewInformation = new RenewInformation();
    }

    /**
     * @return OrderID
     */
    public function getUuid(): OrderID
    {
        return OrderID::fromString($this->uuid);
    }

    /**
     * @return User|null
     */
    public function getUser(): ?User
    {
        return $this->user;
    }

    /**
     * @return Unit
     */
    public function getUnit(): Unit
    {
        return $this->unit;
    }

    /**
     * @return string
     */
    public function getEmail(): string
    {
        return $this->email;
    }

    /**
     * @return string
     */
    public function getDescription(): string
    {
        return $this->description;
    }

    /**
     * @return Subscription|null
     */
    public function getSubscription(): ?Subscription
    {
        return $this->subscription;
    }

    /**
     * @return OrderStatus
     */
    public function getStatus(): OrderStatus
    {
        return new OrderStatus($this->status);
    }

    /**
     * @return string
     */
    public function getToken(): string
    {
        return $this->token;
    }

    /**
     * @return DateTimeImmutable
     */
    public function getCreatedAt(): DateTimeImmutable
    {
        return $this->createdAt;
    }

    /**
     * @return DateTimeImmutable|null
     */
    public function getVerifiedAt(): ?DateTimeImmutable
    {
        return $this->verifiedAt;
    }

    /**
     * @return int|null
     */
    public function getExternalID(): ?int
    {
        return $this->externalID;
    }


    /**
     * @return int
     */
    public function getAmount(): int
    {
        return $this->amount;
    }

    /**
     * @return OrderAction
     */
    public function getAction(): OrderAction
    {
        return new OrderAction($this->action);
    }

    /**
     * @return string
     */
    public function getUrlReturn(): string
    {
        return $this->urlReturn;
    }

    /**
     * @return string
     */
    public function getUrlWebhook(): string
    {
        return $this->urlWebhook;
    }

    /**
     * @return string
     */
    public function getUrlCardInfo(): string
    {
        return $this->urlCardInfo;
    }

    /**
     * @return Collection|Payment[]
     */
    public function getPayments(): Collection
    {
        return $this->payments;
    }

    /**
     * @return RenewInformation
     */
    public function getRenewInformation(): RenewInformation
    {
        return $this->renewInformation;
    }

    /**
     * @return User|null
     */
    public function getVolunteer(): ?User
    {
        return $this->volunteer;
    }

    /**
     * @param User|null $user
     * @return Order
     */
    public function setUser(?User $user): self
    {
        $this->user = $user;
        return $this;
    }

    /**
     * @param Unit $unit
     * @return Order
     */
    public function setUnit(Unit $unit): self
    {
        $this->unit = $unit;
        return $this;
    }

    /**
     * @param Subscription|null $subscription
     * @return Order
     */
    public function setSubscription(?Subscription $subscription): self
    {
        $this->subscription = $subscription;
        return $this;
    }

    /**
     * @param string $email
     * @return Order
     */
    public function setEmail(string $email): self
    {
        $this->email = $email;
        return $this;
    }

    /**
     * @param string $description
     * @return Order
     */
    public function setDescription(string $description): Order
    {
        $this->description = $description;
        return $this;
    }

    /**
     * @param string $status
     * @return Order
     */
    public function setStatus(string $status): self
    {
        $this->status = $status;
        return $this;
    }

    /**
     * @param DateTimeImmutable|null $verifiedAt
     * @return Order
     */
    public function setVerifiedAt(?DateTimeImmutable $verifiedAt): self
    {
        $this->verifiedAt = $verifiedAt;
        return $this;
    }

    /**
     * @param int|null $externalID
     * @return Order
     */
    public function setExternalID(?int $externalID): Order
    {
        $this->externalID = $externalID;
        return $this;
    }

    /**
     * @param int $amount
     * @return Order
     */
    public function setAmount(int $amount): self
    {
        $this->amount = $amount;
        return $this;
    }

    /**
     * @param string $action
     * @return Order
     */
    public function setAction(string $action): self
    {
        $this->action = $action;
        return $this;
    }

    /**
     * @param string $urlReturn
     * @return Order
     */
    public function setUrlReturn(string $urlReturn): self
    {
        $this->urlReturn = $urlReturn;

        return $this;
    }

    /**
     * @param string $urlWebhook
     * @return Order
     */
    public function setUrlWebhook(string $urlWebhook): self
    {
        $this->urlWebhook = $urlWebhook;

        return $this;
    }

    /**
     * @param string $urlCardInfo
     * @return Order
     */
    public function setUrlCardInfo(string $urlCardInfo): Order
    {
        $this->urlCardInfo = $urlCardInfo;
        return $this;
    }

    /**
     * @param string $token
     * @return Order
     */
    public function setToken(string $token): self
    {
        $this->token = $token;

        return $this;
    }

    /**
     * @param RenewInformation $renewInformation
     * @return Order
     */
    public function setRenewInformation(RenewInformation $renewInformation): Order
    {
        $this->renewInformation = $renewInformation;
        return $this;
    }

    public function addPayment(Payment $payment): self
    {

        if (!$this->payments->contains($payment)) {
            $this->payments[] = $payment;
        }
        return $this;
    }

    /**
     * @param User|null $volunteer
     * @return Order
     */
    public function setVolunteer(?User $volunteer): Order
    {
        $this->volunteer = $volunteer;
        return $this;
    }

    public function hasVolunteer(): bool
    {
        return (bool)$this->volunteer;
    }

    public function jsonSerialize()
    {
        return \get_object_vars($this);
    }

    public function getBalanceOperationType(): Type
    {
        $type = Type::PAYMENT_PURPOSE;

        if ($this->getAction()->isHearthAdoption()) {
            if ($this->getSubscription()->isCyclic())
                $type = Type::PAYMENT_HEARTH_ADOPTION_CYCLIC;
            else
                $type = Type::PAYMENT_HEARTH_ADOPTION_ONE_TIME;
        }

        return new Type($type);

//        return new Type($this->getAction()->isHearthAdoption() ? $this->getSubscription()->getKindOfSubscription() : Type::PAYMENT_PURPOSE);
    }

    /**
     * @return int|null
     */
    public function getMonths(): ?int
    {
        return $this->months;
    }

    /**
     * @param int|null $months
     * @return Order
     */
    public function setMonths(?int $months): Order
    {
        $this->months = $months;
        return $this;
    }

    /**
     * @return string
     */
    public function getTransactionNumber(): string
    {
        return $this->transactionNumber;
    }

    /**
     * @param string $transactionNumber
     * @return Order
     */
    public function setTransactionNumber(string $transactionNumber): self
    {
        $this->transactionNumber = $transactionNumber;

        return $this;
    }
}