<?php

namespace App\Controller\Users;

use App\Core\Exception\ApiException;
use App\Entity\Permissions\Permission;
use App\Entity\Users\ReadModel\PersonalDataDTO;
use App\Entity\Users\User;
use App\Models\Users\UserModel;
use App\Services\Users\ActivateAccount;
use App\Services\Users\ResetPassword;
use Nelmio\ApiDocBundle\Annotation\Model;
use OpenApi\Annotations as OA;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\IsGranted;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\ParamConverter;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\Routing\Annotation\Route;

/** @OA\Tag(name="Security") */
#[Route('/api')]
class SecurityController extends AbstractController
{
    public function __construct(
        private ActivateAccount $activateAccount,
        private ResetPassword $resetPassword
    )
    {
    }

    /** @return JsonResponse */
    #[IsGranted(Permission::PUBLIC_ACCESS)]
    #[Route('/login_check', name: 'login_check', methods: ['POST'])]
    public function login(): JsonResponse
    {
        /** @var User $user */
        $user = $this->getUser();

        return $this->json(PersonalDataDTO::fromUser($user));
    }

    /** @OA\Post(summary="Logout") */
    #[IsGranted(Permission::PUBLIC_ACCESS)]
    #[Route('/logout', name: 'logout', methods: ['POST'])]
    public function logout(): JsonResponse
    {
//        throw new \LogicException('This method can be blank - it will be intercepted by the logout key on your firewall.');

        return $this->json("Logout");
    }

    /**
     * @OA\Put(summary="Activate account")
     * @throws ApiException
     */
    #[IsGranted(Permission::PUBLIC_ACCESS)]
    #[Route(path: '/{token}/activate-account', name: 'activate_account', methods: ['PUT'])]
    public function ActivateAccount(User $user): JsonResponse
    {
        $this->activateAccount->activate($user);
        return $this->json('Account has been activated');
    }

    /**
     * @OA\Post(summary="Send reset password token")
     * @OA\RequestBody(
     *     @Model(type=UserModel::class, groups={UserModel::EMAIL})
     * )
     */
    #[IsGranted(Permission::PUBLIC_ACCESS)]
    #[Route(path: '/reset-password', name: 'reset_password_token', methods: ['POST'])]
    #[ParamConverter('userModel', options: ["groups" => [UserModel::EMAIL]], converter: "json_deserialize_param_converter")]
    public function resetPasswordToken(UserModel $userModel): JsonResponse
    {
        $this->resetPassword->sendToken($userModel);
        return $this->json('If it is an account with this address and it is active, an email will be sent to reset your password');
    }

    /**
     * @OA\Put(summary="Reset password")
     * @OA\RequestBody(
     *     @Model(type=UserModel::class, groups={UserModel::PASSWORD})
     * )
     * @throws ApiException
     */
    #[IsGranted(Permission::PUBLIC_ACCESS)]
    #[Route(path: '/{token}/reset-password', name: 'reset_password', methods: ['PUT'])]
    #[ParamConverter('userModel', options: ["groups" => [UserModel::PASSWORD]], converter: "json_deserialize_param_converter")]
    public function resetPassword(User $user, UserModel $userModel): JsonResponse
    {
        $this->resetPassword->resetPassword($user, $userModel);
        return $this->json('Password has been changed.');
    }
}