<?php

namespace App\Controller\Users\Data;

use App\Core\ValidatorModel\ValidatorModelInterface;
use App\Entity\Permissions\Permission;
use App\Entity\Reports\ReadModel\ReportsQueryInterface;
use App\Entity\Users\User;
use App\Models\Reports\RequestDatesModel;
use OpenApi\Annotations as OA;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\IsGranted;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\ParamConverter;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Annotation\Route;

/**
 * @OA\Tag(name="Users")
 */
#[Route('/api/users', name: 'users_')]
class UserReportDataController extends AbstractController
{
    public function __construct(
        private ReportsQueryInterface $reportsQuery,
        private ValidatorModelInterface $validator
    )
    {
    }

    /**
     * @OA\Get(summary="Get finances report data")
     * @OA\Parameter(name="year", in="query", description="date start")
     * @param Request $request
     * @param User $user
     * @return JsonResponse
     */
    #[IsGranted(Permission::ACCOUNT_DATA_MANAGEMENT)]
    #[IsGranted(Permission::PRIVATE_ACCESS, 'user')]
    #[Route(path: '/{userId}/finances', name: 'finances', methods: ['GET'])]
    #[ParamConverter('user', options: ['mapping' => ['userId' => 'uuid']])]
    public function financesReport(Request $request, User $user): JsonResponse
    {
        $year = $request->query->get('year');
        
        $requestDates = new RequestDatesModel(year: $year);
        $this->validator->validate($requestDates);

        return $this->json(
            $this->reportsQuery->getYearReportDataByUserID($user->getUuid(), $requestDates->getYear())
        );
    }
}