<?php

namespace App\Controller\Orders;

use App\Core\ValidatorModel\ValidatorModelInterface;
use App\Entity\Permissions\Permission;
use App\Entity\Units\Unit;
use App\Entity\Users\User;
use App\Models\Orders\OrderPurposeModel;
use App\Services\Orders\CreateOrderForPurposePayment;
use OpenApi\Annotations as OA;
use Nelmio\ApiDocBundle\Annotation\Model;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\IsGranted;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\ParamConverter;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\Serializer\SerializerInterface;

/**
 * @OA\Tag(name="Orders purpose")
 */
#[Route(path: '/api/orders', name: 'orders_')]
class PurposeOrderController extends AbstractController
{
    public function __construct(
        private SerializerInterface $serializer,
        private ValidatorModelInterface $validatorModel,
        private CreateOrderForPurposePayment $createOrderForPurposePayment
    )
    {
    }

    /**
     * @OA\Post(summary="Make new payments for unit purpose")
     * @OA\RequestBody(
     *     @Model(type=OrderPurposeModel::class)
     * )
     * @OA\Response(response=200, description="Return url to redirect")
     * @param Request $request
     * @param Unit $unit
     * @return Response
     */
    #[IsGranted(Permission::PUBLIC_ACCESS)]
    #[Route('/{unitID}/new-order', name: 'new_purpose', methods: ['POST'])]
    #[ParamConverter('unit', options: ['mapping' => ['unitID' => 'uuid']])]
    public function purposeOrder(Request $request, Unit $unit): Response
    {
        /** @var User|null $user */
        $user = $this->getUser();

        /** @var OrderPurposeModel $orderPurposeModel */
        $orderPurposeModel = $this->serializer->deserialize($request->getContent(), OrderPurposeModel::class, 'json');
        $orderPurposeModel->setUser($user);

        $this->validatorModel->validate(value: $orderPurposeModel, groups: $orderPurposeModel->getValidationGroups());

        $redirectUrl = $this->createOrderForPurposePayment->purposeOrder($orderPurposeModel, $unit);

        return $this->json(['redirectURL' => $redirectUrl]);
    }
}