<?php

namespace App\Adapter\Users\ReadModel;

use App\Core\Paginator\ObjectValue\PaginationRequest;
use App\Core\Paginator\PaginatorInterface;
use App\Entity\BalanceOperations\Helper\Type;
use App\Entity\Roles\Role;
use App\Entity\Users\Helper\VolunteerBalanceHistoryDTO;
use App\Entity\Users\ReadModel\VolunteerBalanceDTO;
use App\Entity\Users\ReadModel\VolunteerDTO;
use App\Entity\Users\ReadModel\VolunteersBalanceQueryInterface;
use App\Entity\Users\UserID;
use Doctrine\DBAL\Connection;

class VolunteersBalanceQuery implements VolunteersBalanceQueryInterface
{
    public function __construct(
        private Connection $connection,
        private PaginatorInterface $paginator
    )
    {
    }

    public function getVolunteerBalanceHistoryByUserID(UserID $userID, PaginationRequest $paginationRequest): array
    {
        $qb = $this->connection->createQueryBuilder();

        $query = $qb
            ->select('b.uuid as balanceOperationID')
            ->addSelect('b.title as title')
            ->addSelect('b.amount as amount')
            ->addSelect('b.balance as balance')
            ->addSelect('b.created_at as createdAt')
            ->addSelect('b.type as type')
            ->from('balance_operation', 'b')
//            ->leftJoin('b','sysOrder', 'o', 'b.basic_order_id = o.uuid')
//            ->leftJoin('o', 'unit', 'u', 'o.unit_id = u.uuid')
            ->where('b.user_id = :userID')
            ->orderBy('b.created_at', 'DESC')
            ->setParameter('userID', $userID, 'uuid');

        $this->paginator->addPagination($query, $paginationRequest, true);
        $results = $query->execute()->fetchAllAssociative();

        return array_map(fn($data) => VolunteerBalanceHistoryDTO::fromArray($data), $results);

    }

    public function getVolunteerBalanceByUserID(UserID $userID): VolunteerBalanceDTO
    {
        $qb = $this->connection->createQueryBuilder();

        $result = $qb
            ->select('SUM(b.amount) as totalOperationsCount')
            ->addSelect('SUM(sO.amount) as totalAmountCount')
            ->addSelect('SUM(CASE WHEN b.type = :adminOperation THEN b.amount ELSE 0 END) as adminOperationsCount')
            ->addSelect('SUM(CASE WHEN b.type != :adminOperation THEN b.amount ELSE 0 END) as paymentOperationsCount')
            ->from('balance_operation', 'b')
            ->leftJoin('b', 'sysOrder', 'sO', 'b.basic_order_id = sO.uuid')
            ->where('b.user_id = :userID')
            ->setParameter('userID', $userID, 'uuid')
            ->setParameter('adminOperation', Type::ADMIN_OPERATION)
            ->execute()
            ->fetch();

        return VolunteerBalanceDTO::fromArray($result);
    }


    public function getVolunteers(PaginationRequest $paginationRequest): array
    {
        $qb = $this->connection->createQueryBuilder();

        $query = $qb
            ->select('SUM(b.amount) as totalOperationsCount')
            ->addSelect('SUM(sO.amount) as totalAmountCount')
            ->addSelect('SUM(CASE WHEN b.type = :adminOperation THEN b.amount ELSE 0 END) as adminOperationsCount')
            ->addSelect('SUM(CASE WHEN b.type != :adminOperation THEN b.amount ELSE 0 END) as paymentOperationsCount')
            ->addSelect('u.uuid as volunteerId')
            ->addSelect('u.name as name')
            ->addSelect('u.surname as surname')
            ->addSelect('u.email as email')
            ->addSelect('u.reference_code as referenceCode')
            ->from('user', 'u')
            ->leftJoin('u', 'balance_operation', 'b', 'u.uuid = b.user_id')
            ->leftJoin('b', 'sysOrder', 'sO', 'b.basic_order_id = sO.uuid')
            ->leftJoin('u', 'role', 'r', 'u.role_id = r.uuid')
            ->groupBy('u.uuid')
            ->where('r.original_name = :volunteer')
            ->setParameter('adminOperation', Type::ADMIN_OPERATION)
            ->setParameter('volunteer', Role::VOLUNTEER);

        $this->paginator->addPagination($query, $paginationRequest, true);
        $results = $query->execute()->fetchAllAssociative();

        return array_map(fn($data) => VolunteerDTO::fromArray($data), $results);
    }

    public function getVolunteerByReferenceCode(string $referenceCode): ?VolunteerDTO
    {
        $qb = $this->connection->createQueryBuilder();

        $result = $qb
            ->select('u.uuid as volunteerId')
            ->addSelect('u.name')
            ->addSelect('u.surname')
            ->addSelect('u.email')
            ->addSelect('u.reference_code as referenceCode')
            ->from('user', 'u')
            ->leftJoin('u', 'role', 'r', 'u.role_id = r.uuid')
            ->andWhere('u.reference_code = :referenceCode')
            ->andWhere('r.original_name = :volunteer')
            ->setParameter('referenceCode', $referenceCode)
            ->setParameter('volunteer', Role::VOLUNTEER)
            ->execute()
            ->fetch();

        return $result ? VolunteerDTO::fromArray($result) : null;
    }
}