<?php

namespace App\Adapter\Users\ReadModel;

use App\Core\Paginator\ObjectValue\PaginationRequest;
use App\Core\Paginator\PaginatorInterface;
use App\Entity\Orders\Helper\OrderStatus;
use App\Entity\Users\ReadModel\PersonalDataDTO;
use App\Entity\Users\ReadModel\UserDTO;
use App\Entity\Users\ReadModel\UsersQueryInterface;
use App\Entity\Users\UserID;
use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Exception;
use Doctrine\DBAL\Driver\Exception as DriverException;

class UsersQuery implements UsersQueryInterface
{
    public function __construct(
        private Connection $connection,
        private PaginatorInterface $paginator
    )
    {
    }

    /**
     * @throws Exception
     * @throws DriverException
     * @throws \ReflectionException
     */
    public function findAllUsers(PaginationRequest $paginationRequest): array
    {
        $qb = $this->connection->createQueryBuilder();

        $query = $qb
            ->select('u.uuid as userId')
            ->addSelect('u.name')
            ->addSelect('u.surname')
            ->addSelect('u.email')
            ->addSelect('r.uuid as roleId')
            ->addSelect('r.name as roleName')
            ->addSelect('u.is_active as isActive')
            ->addSelect('u.is_blocked as isBlocked')
            ->addSelect('SUM(p.amount) as paymentSum')
            ->from('user', 'u')
            ->leftJoin('u', 'role', 'r', 'u.role_id = r.uuid')
            ->leftJoin('u', 'sysOrder', 'sO', 'u.uuid = sO.user_id AND sO.status = :orderStatus')
            ->leftJoin('sO', 'payment', 'p', 'sO.uuid = p.order_id AND p.created_at <= :currentDate')
            ->groupBy('u.uuid')
            ->setParameter('orderStatus', OrderStatus::VERIFIED)
            ->setParameter('currentDate', new \DateTimeImmutable(), 'datetime_immutable');

        $this->paginator->addPagination($query, $paginationRequest, true);
        $results = $query->execute()->fetchAllAssociative();

        return array_map(fn($data) => UserDTO::fromArray($data), $results);
    }

    public function findOneByUserID(UserID $userID): ?PersonalDataDTO
    {
        $qb = $this->connection->createQueryBuilder();

        $result = $qb
            ->select('u.uuid ')
            ->addSelect('u.name')
            ->addSelect('u.surname')
            ->addSelect('u.email')
            ->addSelect('u.reference_code as referenceCode')
            ->from('user', 'u')
            ->where('u.uuid = :userID')
            ->setParameter('userID', $userID, 'uuid')
            ->execute()
            ->fetch();

        return $result ? PersonalDataDTO::fromArray($result) : null;
    }
}