<?php

namespace App\Adapter\Reports\ReadModel;

use App\Core\Doctrine\SubQueryBuilder;
use App\Core\Paginator\ObjectValue\PaginationRequest;
use App\Core\Paginator\PaginatorInterface;
use App\Entity\Agreement\Agreement;
use App\Entity\Orders\Helper\OrderStatus;
use App\Entity\Orders\ReadModel\OrdersSubQueryInterface;
use App\Entity\Reports\ReadModel\AnnualReportOrderDTO;
use App\Entity\Reports\ReadModel\FinanceReportDTO;
use App\Entity\Reports\ReadModel\OrderReportDTO;
use App\Entity\Reports\ReadModel\QuarterReportDTO;
use App\Entity\Reports\ReadModel\QuarterReportRecipientDTO;
use App\Entity\Reports\ReadModel\ReportsQueryInterface;
use App\Entity\Reports\ReadModel\ReportsSubQueryInterface;
use App\Entity\Reports\ReadModel\SubscriptionReportDTO;
use App\Entity\Reports\ReadModel\YearReportDTO;
use App\Entity\Reports\Report;
use App\Entity\Users\UserID;
use App\Models\Reports\RequestDatesModel;
use Doctrine\DBAL\Connection;
use Symfony\Component\Serializer\Normalizer\NormalizerInterface;
use function Doctrine\DBAL\Query\QueryBuilder;

class ReportsQuery implements ReportsQueryInterface
{
    public function __construct(
        private Connection $connection,
        private OrdersSubQueryInterface $ordersSubQuery,
        private NormalizerInterface $normalizer,
        private ReportsSubQueryInterface $reportsSubQuery,
        private PaginatorInterface $paginator
    )
    {
    }

    public function getFinanceReportData(RequestDatesModel $requestDatesModel): array
    {
        $qb = $this->connection->createQueryBuilder();

        $results = $qb
            ->select('u.type')
            ->addSelect('u.title')
            ->addSelect('p.money as moneyPurpose')
            ->addSelect('sO.collectedMoney as collectedMoney')
            ->addSelect('a.children')
            ->addSelect('s.supportedChildren as supportedChildren')
            ->addSelect('s.supportedChildrenMoney as supportedChildrenMoney')
            ->addSelect('o.totalMoney as totalMoney')
            ->from('unit', 'u')
            ->leftJoin('u', 'purpose', 'p', 'u.purpose_id = p.uuid')
            ->leftJoin('u', 'hearth_adoption', 'a', 'u.adoption_id = a.uuid')
            ->leftJoin('u', SubQueryBuilder::prepare($this->ordersSubQuery->collectedMoneyQuery(), $qb), 'sO', 'u.uuid = sO.unit_id')
            ->leftJoin('u', SubQueryBuilder::prepare($this->ordersSubQuery->supportedChildrenQuery(), $qb), 's', 'u.uuid = s.unit_id')
            ->leftJoin('u', SubQueryBuilder::prepare($this->ordersSubQuery->ordersAmount($requestDatesModel->getDateStart(), $requestDatesModel->getDateEnd()), $qb), 'o', 'u.uuid = o.unit_id')
            ->execute()
            ->fetchAllAssociative();

        return array_map(fn($data) => FinanceReportDTO::fromArray($data), $results);
    }

    public function getSubscriptionsReportData(): array
    {
        $qb = $this->connection->createQueryBuilder();

        $results = $qb
            ->select('s.uuid as subscriptionId')
            ->addSelect('u.uuid as unitId')
            ->addSelect('u.title as unitTitle')
            ->addSelect('u.type as unitType')
            ->addSelect('uS.uuid as userId')
            ->addSelect('uS.name as userName')
            ->addSelect('uS.surname as userSurname')
            ->addSelect('uS.email as userEmail')
            ->addSelect('v.reference_code as referenceCode')
            ->addSelect('s.kind_of_subscription as kindOfSubscription')
            ->addSelect('s.created_at as createdAt')
            ->addSelect('s.date_expiration as dateExpiration')
            ->addSelect('s.date_end as dateEnd')
            ->addSelect('s.reason_for_closure as reason')
            ->addSelect('s.monthly_cost as monthlyCost')
            ->addSelect('s.children as children')
            ->from('subscription', 's')
            ->leftJoin('s', 'unit', 'u', 's.unit_id = u.uuid')
            ->leftJoin('s', 'user', 'uS', 's.user_id = uS.uuid')
            ->leftJoin('s', 'user', 'v', 's.volunteer_id = v.uuid')
            ->execute()
            ->fetchAllAssociative();

        return array_map(fn($data) => $this->normalizer->normalize(SubscriptionReportDTO::fromArray($data)), $results);
    }

    public function getOrdersReportData(RequestDatesModel $requestDates): array
    {
        $qb = $this->connection->createQueryBuilder();

        $query = $qb
            ->select('sO.uuid as orderId')
            ->addSelect('sO.unit_id as unitId')
            ->addSelect('u.title as unitTitle')
            ->addSelect('us.uuid as userId')
            ->addSelect('us.name as userName')
            ->addSelect('us.surname as userSurname')
            ->addSelect('sO.email as email')
            ->addSelect('v.reference_code as referenceCode')
            ->addSelect('(CASE WHEN sO.subscription_id IS NULL THEN sO.action ELSE s.kind_of_subscription END) as status')
            ->addSelect('sO.verified_at as verifiedAt')
            ->addSelect('sO.amount as amount')
            ->addSelect('(CASE WHEN uCA.agreement_id IS NULL THEN FALSE ELSE TRUE END) as hasCommercialAgreement')
            ->addSelect('sO.months as months')
            ->addSelect('sO.transaction_number as transactionNumber')
            ->from('sysOrder', 'sO')
            ->leftJoin('sO', 'unit', 'u', 'sO.unit_id = u.uuid')
            ->leftJoin('sO', 'user', 'us', 'sO.user_id = us.uuid')
            ->leftJoin('us', SubQueryBuilder::prepare($this->reportsSubQuery->usersAgreement(Agreement::COMMERCIAL_INFORMATION), $qb), 'uCA', 'us.uuid = uCA.user_id')
            ->leftJoin('sO', 'user', 'v', 'sO.volunteer_id = v.uuid')
            ->leftJoin('sO', 'subscription', 's', 'sO.subscription_id = s.uuid')
            ->andWhere('sO.status = :orderStatus')
            ->setParameter('orderStatus', OrderStatus::VERIFIED);

        if ($requestDates->getDateStart()) {
            $query
                ->andWhere('DATE(sO.verified_at) >= :dateStart')
                ->setParameter('dateStart', $requestDates->getDateStart()->format('Y-m-d'));
        }
        if ($requestDates->getDateEnd()) {
            $query
                ->andWhere('DATE(sO.verified_at) <= :dateEnd')
                ->setParameter('dateEnd', $requestDates->getDateEnd()->format('Y-m-d'));
        }
        $results = $query
            ->execute()
            ->fetchAllAssociative();

        return array_map(fn($data) => $this->normalizer->normalize(OrderReportDTO::fromArray($data)), $results);
    }

    public function getYearReportDataByUserID(UserID $userID, ?int $year = null): array
    {
        $qb = $this->connection->createQueryBuilder();

        $results = $qb
            ->addSelect('u.uuid as unitId')
            ->addSelect('u.title as unitTitle')
            ->addSelect('o.totalMoney as totalMoney')
            ->from('unit', 'u')
            ->leftJoin('u', SubQueryBuilder::prepare($this->ordersSubQuery->ordersAmountByUserAndYear($userID, $year), $qb), 'o', 'u.uuid = o.unit_id')
            ->andWhere('o.totalMoney IS NOT NULL')
            ->execute()
            ->fetchAllAssociative();

        return array_map(fn($data) => YearReportDTO::fromArray($data), $results);
    }

    public function getQuartersReportData(PaginationRequest $paginationRequest): array
    {
        $qb = $this->connection->createQueryBuilder();

        $query = $qb
            ->addSelect('u.uuid as unitID')
            ->addSelect('u.title as unitTitle')
            ->addSelect('r.date_start as dateStart')
            ->addSelect('r.date_end as dateEnd')
            ->addSelect('r.created_at as createdAt')
            ->addSelect('r.description as description')
            ->addSelect('r.post_id as postID')
            ->from('report', 'r')
            ->leftJoin('r', 'unit', 'u', 'r.unit_id = u.uuid');

        $this->paginator->addPagination($query, $paginationRequest);
        $results = $query->execute()->fetchAllAssociative();

        return array_map(fn($data) => QuarterReportDTO::fromArray($data), $results);
    }

    public function getQuarterReportRecipients(Report $report): array
    {
        $qb = $this->connection->createQueryBuilder();

        $query = $qb
            ->addSelect('us.email as email')
            ->addSelect('us.name as userName')
            ->addSelect('us.surname as userSurname')
            ->from('payment', 'p')
            ->leftJoin('p', 'sysOrder', 'sO', 'p.order_id = sO.uuid')
            ->leftJoin('sO', 'user', 'us', 'sO.user_id = us.uuid')
            ->leftJoin('us', SubQueryBuilder::prepare($this->reportsSubQuery->usersAgreement(Agreement::QUARTERLY_REPORT_SUPPORTED_UNIT), $qb), 'uCA', 'us.uuid = uCA.user_id')
            ->andWhere('sO.unit_id = :unitID')
            ->andWhere('sO.status = :status')
            ->andWhere('DATE(p.created_at) BETWEEN DATE(:dateStart) AND DATE(:dateEnd)')
            ->andWhere('sO.user_id IS NOT NULL')
            ->andWhere('uCA.agreement_id IS NOT NULL')
            ->addGroupBy('us.email, uCA.agreement_id')
            ->setParameter('unitID', $report->getUnit()->getUuid(), 'uuid')
            ->setParameter('status', OrderStatus::VERIFIED)
            ->setParameter('dateStart', $report->getDateStart()->format('Y-m-d'))
            ->setParameter('dateEnd', $report->getDateEnd()->format('Y-m-d'));
        $results = $query->execute()->fetchAllAssociative();

        return array_map(fn($data) => QuarterReportRecipientDTO::fromArray($data), $results);
    }

    public function getAnnualReportDataByUserID(UserID $userID, int $year): array
    {
        $qb = $this->connection->createQueryBuilder();

        $query = $qb
            ->addSelect('u.title as unitTitle')
            ->addSelect('sO.transaction_number as transactionNumber')
            ->addSelect('sO.verified_at as verifiedAt')
            ->addSelect('sO.amount as amount')
            ->from('sysOrder', 'sO')
            ->leftJoin('sO', 'unit', 'u', 'sO.unit_id = u.uuid')
            ->andWhere('sO.status = :orderStatus')
            ->andWhere('sO.user_id = :userID')
            ->andWhere('YEAR(sO.verified_at) = :year')
            ->addOrderBy('u.title', 'ASC')
            ->addOrderBy('sO.verified_at', 'ASC')
            ->setParameter('userID', $userID, 'uuid')
            ->setParameter('year', $year)
            ->setParameter('orderStatus', OrderStatus::VERIFIED);


        $results = $query
            ->execute()
            ->fetchAllAssociative();

        return array_map(fn($data) => AnnualReportOrderDTO::fromArray($data), $results);
    }
}